import datetime
import pandas as pd
import MetaTrader5 as mt5
import ta
import time


buy = mt5.ORDER_TYPE_BUY
sell = mt5.ORDER_TYPE_SELL


def create_order(symbol, lot, order_type, sl=0.0, tp=0.0, comment='hashem'):
    symbol_info = mt5.symbol_info(symbol)
    
    filling_mode = symbol_info.filling_mode
    if filling_mode == 1:
        filling_mode = mt5.ORDER_FILLING_FOK
    elif filling_mode == 2:
        filling_mode = mt5.ORDER_FILLING_IOC
    else:
        filling_mode = mt5.ORDER_FILLING_FOK 
    
    price_info = mt5.symbol_info_tick(symbol)
    price = price_info.ask if order_type == buy else price_info.bid

    request = {
        "action": mt5.TRADE_ACTION_DEAL,
        "symbol": symbol,
        "volume": lot,
        "type": order_type,
        "price": price,
        "sl": sl,
        "tp": tp,
        "deviation": 20,
        "comment": comment,
        "type_time": mt5.ORDER_TIME_GTC,
        "type_filling": filling_mode,
    }
    
    result = mt5.order_send(request)
    if result.retcode != mt5.TRADE_RETCODE_DONE:
        print(f"خطا در ثبت سفارش: {result.comment}")
    
    return result
           

def close_order(ticket):

    position = mt5.positions_get(ticket=ticket)
    if position is None or len(position) == 0:
        print(f"پوزیشن {ticket} پیدا نشد")
        return
    
    position = position[0]
    symbol = position.symbol
    volume = position.volume
    
    close_type = mt5.ORDER_TYPE_SELL if position.type == mt5.POSITION_TYPE_BUY else mt5.ORDER_TYPE_BUY
    
    price_info = mt5.symbol_info_tick(symbol)
    price = price_info.bid if position.type == mt5.POSITION_TYPE_BUY else price_info.ask
    
    filling_modes = [mt5.ORDER_FILLING_FOK, mt5.ORDER_FILLING_IOC, mt5.ORDER_FILLING_RETURN]
    
    for filling_mode in filling_modes:
        request = {
            "action": mt5.TRADE_ACTION_DEAL,
            "symbol": symbol,
            "volume": volume,
            "type": close_type,
            "position": ticket,
            "price": price,
            "deviation": 20,
            "magic": 0,
            "comment": "Close position",
            "type_time": mt5.ORDER_TIME_GTC,
            "type_filling": filling_mode,
        }
        
        mt5.order_send(request)
     

def close_all_positions():
    positions = mt5.positions_get()
    if positions is None:
        pass

    for position in positions:
        p_ticket = position._asdict()['ticket']
        close_order(p_ticket)

   
def close_half_positions():
    positions = mt5.positions_get()
    if positions is None or len(positions) == 0:
        return

    half_count = len(positions) // 2

    for i, position in enumerate(positions):
        if i >= half_count:
            break
        p_ticket = position._asdict()['ticket']
        
        close_order(p_ticket)


def total_positons():
    positions_total=mt5.positions_total()
    return positions_total


def balance():
    balance = mt5.account_info().balance
    return balance


def profit():
    positions = mt5.positions_get()
    profit = 0
    for position in positions:
        profit += position.profit
    return profit


def candle(symbol='XAUUSD', tf='3m', limit=100):
 
    tf = int(tf.replace('m', ''))
    total_tf = tf * limit * 2
    
    raw_data = mt5.copy_rates_from_pos(symbol, mt5.TIMEFRAME_M1, 0, total_tf)
    if raw_data is None or len(raw_data) == 0:
        return pd.DataFrame()
    
    df = pd.DataFrame(raw_data, columns=['time', 'open', 'high', 'low', 'close', 'tick_volume'])
    df['time'] = pd.to_datetime(df['time'], unit='s')
    df['aligned_time'] = df['time'].dt.floor(f'{tf}min')
    
    resampled = df.groupby('aligned_time').agg({
        'open': 'first',
        'high': 'max',
        'low': 'min',
        'close': 'last',
        'tick_volume': 'sum'
    }).reset_index()
    
    result = (resampled
             .sort_values('aligned_time', ascending=False)
             .head(limit)
             .sort_values('aligned_time')
             .set_index('aligned_time'))
    
    return result.iloc


def heikin_ashi(symbol='XAUUSD', tf='3m', limit=100):
    tf = int(tf.replace('m', ''))
    total_tf = tf * limit * 2
    
    raw_data = mt5.copy_rates_from_pos(symbol, mt5.TIMEFRAME_M1, 0, total_tf)
    if raw_data is None or len(raw_data) == 0:
        return pd.DataFrame()
    
    df = pd.DataFrame(raw_data, columns=['time', 'open', 'high', 'low', 'close', 'tick_volume'])
    df['time'] = pd.to_datetime(df['time'], unit='s')
    df['aligned_time'] = df['time'].dt.floor(f'{tf}min')
    
    resampled = df.groupby('aligned_time').agg({
        'open': 'first',
        'high': 'max',
        'low': 'min',
        'close': 'last',
        'tick_volume': 'sum'
    }).reset_index()
    
    df = (resampled
          .sort_values('aligned_time', ascending=False)
          .head(limit)
          .sort_values('aligned_time')
          .set_index('aligned_time'))
    
    ha_df = pd.DataFrame(index=df.index)
    ha_df['close'] = (df['open'] + df['high'] + df['low'] + df['close']) / 4
    ha_df['open'] = 0.0
    ha_df.iloc[0, ha_df.columns.get_loc('open')] = df['open'].iloc[0]
    
    for i in range(1, len(df)):
        ha_df.iloc[i, ha_df.columns.get_loc('open')] = (
            ha_df['open'].iloc[i-1] + ha_df['close'].iloc[i-1]
        ) / 2
    
    ha_df['high'] = ha_df[['open', 'close']].max(axis=1).combine(df['high'], max)
    ha_df['low'] = ha_df[['open', 'close']].min(axis=1).combine(df['low'], min)
    
    return ha_df.iloc


def check_candle(symbol , tf , cdl = -1 , candle_type = 'ca'):
    
    if candle_type == 'ca':
        ohlc = candle(symbol, tf)
    else:
        ohlc = heikin_ashi(symbol, tf)

    if ohlc[cdl]['open'] > ohlc[cdl]['close']:
        return 'short'
    else:
        return 'long'
    

def isBeta(symbol , tf , index = -1 ):
    kandels = candle(symbol, tf)
    res = kandels[index]
    if res['open'] > res['close']:
        # short kandel
        if res['open'] == res['high'] :
            return True
        else:
            return False
        
    elif res['open'] < res['close']:
        # long kandel
        if res['open'] == res['low'] :
            return True
        else:
            return False
    else:
        return False
    

def isBack(symbol , tf , index = -1 , upOrDown = 'up' ):
    candles = candle(symbol, tf)
    res = candles[index]
    if res['open'] > res['close']:
        # short 
        if upOrDown == 'up'and (res['open'] - res['close'])*2 < res['high'] - res['open'] and res['close'] - res['low'] < res['open'] - res['close']:
            return True
        elif upOrDown == 'down'and (res['open'] - res['close'])*3 < res['close'] - res['low'] and  res['high'] - res['open'] < res['open'] - res['close']:
            return True
        else:
            return False
        
    elif res['open'] < res['close']:
        # long 
        if upOrDown == 'up'and (res['close'] - res['open'])*3 < res['high'] - res['close'] and res['close'] - res['open'] > res['open'] - res['low'] :
            return True
        
        elif upOrDown == 'down'and (res['close'] - res['open'])*2 < res['open'] - res['low'] and res['close'] - res['open'] > res['high'] - res['close']   :
            return True
        
        else:
            return False
    else:
        return False


def body(symbol, tf , index = -1):
    candles = candle(symbol, tf)
    res = candles[index]
    if res['open'] > res['close']:
        # short
        body = res['open'] - res['close']
        return body
        
    elif res['open'] < res['close']:
        # long 
        body = res['close'] - res['open']
        return body
    else:
        return 0
    

def isgap(symbol, tf):
    candles = candle(symbol, tf)
    #long
    if candles[-2]['open'] > candles[-3]['close'] and check_candle(symbol, tf , -2) == 'long' and check_candle(symbol, tf , -3) == 'long':
        return True
    #short
    if candles[-2]['open'] < candles[-3]['close'] and check_candle(symbol, tf , -2) == 'short' and check_candle(symbol, tf , -3) == 'short':
        return True
    else:
        return False
    

def check_time(start_hour, end_hour):
    current_time = datetime.datetime.now(datetime.UTC).time()
    if current_time.hour >= start_hour and current_time.hour <= end_hour:
        return True
    else:
        return False
    

def modify_stop(ticket, new_stop_loss):

    position = mt5.positions_get(ticket=ticket)
    if not position:
        return False
    
    position = position[0]

    request = {
        "action": mt5.TRADE_ACTION_SLTP,
        "position": position.ticket,
        "sl": new_stop_loss,
        "tp": position.tp,
        "symbol": position.symbol,
        "type": position.type,
        "volume": position.volume,
    }

    result = mt5.order_send(request)


def modify_tp(ticket, new_tp):

    position = mt5.positions_get(ticket=ticket)
    if not position:
        return False
    
    position = position[0]

    request = {
        "action": mt5.TRADE_ACTION_SLTP,
        "position": position.ticket,
        "sl": position.sl,
        "tp": new_tp,
        "symbol": position.symbol,
        "type": position.type,
        "volume": position.volume,
    }

    result = mt5.order_send(request)


